/*
** ADDA.P
**
** Ansprache der Kolter PCI-ADDAC4
** ber den Winston von Pearl90 aus
**
** Globale Prozeduren/Funktionen:
**
**    INIT_ADDA: PROC;
**          Initialisierung der ADDA.
**
**    DacOut: PROC( kanal FIXED(31) , spannung FLOAT ) GLOBAL ;
**          kanal: gibt den Kanal an, ueber den spannung ausgegeben werden soll.
**                 Zulaessige Werte: DACA, DACB, DACC oder DACD      
**          spannung: auszugebende Spannung -10. ... +10.
**   
**    ADread: PROC( kanal FIXED ) RETURNS( FLOAT ) GLOBAL ;
**          kanal gibt die Nummer des A/D-Kanals an dessen anliegende Spannung 
**          zurueckgegeben wird.
**          Zulaessig sind die Werte 0...15, bei differentiellen Eingaengen 
**          sind nur Werte 0 ... 7 sinnvoll.
**
** Tasks:
**    
**   TestAD: zyklisches Auslesen aller A/D-Kanaele und Ausgabe der gemessenen
**           Spannungen
**
**   TestDA: Ausgaben von Testmustern ueber alle D/A-Kanaele
**           DAC A: steigender Saegezahn
**           DAC B: fallender Saegezahn
**           DAC C: Dreieck
**           DAC D: Sinus
**
**   InitADDA: Initialisiert die AD-DA-Karte bzw. den Treiber
**
**   StopADDA: beendet die Ausgaben von TestAD und TestDA
**
**   Tick: Togglet den Digitalausgang der ADDA
**         Vor Benutzung muss InitADDA aufgerufen werden!
**
**   Square: Gibt ein Rechtecksignal ueber den Digitalausgang aus
**
**   vout kanal spannung 
**       Ausgabe einer Spannung auf dem D/A-Teil
**       kanal:    1...4 entsprechend DAC A ... DAC D
**       spannung: +10. ... -10. in Volt'
**       vout ist als Shellbefehl realisiert und kann z.B. als 
**       'vout 1, 5.5' aufgerufen werden.
**
** (c) IEP GmbH Langenhagen, 2004
*/

#DEFINE THIS_IS_ADDA = 1 

/*+T*/
/*+M*/
SC=$3000;
SHELLMODULE ADDA;
   Vout: 'vout' ;

PROBLEM ;

   SPC TY DATION INOUT ALPHIC GLOBAL;
#INCLUDE KlibDrv.H 
#INCLUDE Klib.H 
#INCLUDE ../INC/SYSSBR.H 
#INCLUDE ADDA.H

   DCL Kolter_VendorID     INV BIT(16) INIT( '1001'B4 ) ;
   DCL PCI_ADDA_DeviceID   INV BIT(16) INIT( '0012'B4 ) ;

   DCL Driver        PCI_DRIVER ;
   DCL Basisadresse  FIXED(31) INIT( 0(31) ) ;
   DCL DriverHandle  FIXED(31) INIT( 0(31) ) ;
   DCL semaAD        SEMA ;   /* synchronisiert den Zugriff auf den A/D-Teile */
   DCL semaDA        SEMA ;   /* synchronisiert den Zugriff auf den D/A-Teil  */
   DCL BeendeADDA    BIT(1)    INIT('1'B ) ;
   /* 
   ** DACs - Spannungsausgaenge -10. ... +10. Volt
   **
   ** Die DACs belegen 2 Adressen auf dem Bus.
   **
   ** Die Offsets fuer DACA ... DACD sind in ADDA.H definiert
   ** LSB: Karten-Basisadresse + Offset
   ** MSB: Karten-Basisadresse + Offset + 4
   ** 
   ** Die DACs uebernehmen die Ausgabedaten beim Lesen von
   ** Latch: Karten-Basisadresse + 29
   */
   DCL DAC_LATCH        INV FIXED(31) INIT( 29(31) ) ; /* Offset fuer Latch-Impuls  */

   /*
   ** ADC
   **
   ** Der ADC besitzt einen Multiplexer. Der zu wandelnde Kanal wird                
   ** ueber AD_MUX eingestellt, nach Umschaltung sind bis zu 10 us Einschwingzeit
   ** erforderlich.
   ** Nach dem Einschwingen des Eingangsverstaerkers wird ueber das LS-Bit der
   ** Read/Convert-Adresse eine Wandlung gestartet. Das Ende der Wandlung
   ** kann daran erkannt werden, dass das LSB der Ready-Adresse auf 1 geht.
   */
   DCL AD_MUX           INV FIXED(31) INIT(    4(31) ) ; /* 0...15, Kanalwahl         */
   DCL AD_RC            INV FIXED(31) INIT(    0(31) ) ; /* 1/0: Read/Convert-Impuls  */
      DCL AD_READ_MSB   INV FIXED( 7) INIT(    1( 7) ) ; /* Read+LSB lesen            */
      DCL AD_READ_LSB   INV FIXED( 7) INIT(    3( 7) ) ; /* Read+MSB lesen            */
      DCL AD_CONVERT    INV FIXED( 7) INIT(    0( 7) ) ; /* Wandlung starten          */
   DCL AD_READY         INV FIXED(31) INIT(    1(31) ) ; /* LSb == 1/0: ready/busy    */

   DCL AD_RESOLUTION    INV FIXED(31) INIT( 4095(31) ) ; /* max. Wandlerwert          */
   DCL AD_VOLT_RANGE    INV FLOAT     INIT(   20.    ) ; /* Spannungsbereich          */
   DCL AD_WW_NULL       INV FIXED(31) INIT( 2048(31) ) ; /* Wandlerwert fuer 0. Volt  */
   
   /*
   ** DIO
   **
   ** Es steht jeweils ein Bit Digital-In und Digital-out zur Verfuegung
   */
   DCL DIG_IN           INV FIXED(31) INIT(   17(31) ) ; /* Bit 1 ist Input           */
   DCL DIG_OUT          INV FIXED(31) INIT(   16(31) ) ; /* Bit 0 ist Output          */

   /* Eingangskanal merken, um ggf. schalten des Multiplexers zu vermeiden */
   DCL AD_LetzterKanal      FIXED( 7) INIT(  0( 7) ) ; 
   DCL AD_lsb_spannung      FLOAT                    ;


INIT_ADDA: PROC GLOBAL; /*..........................................*/
   /* Die A/D-Kanaele initialisieren                                */
   DCL find       PCI_FIND_STRUCT ;
   IF Basisadresse EQ 0(31) THEN
      CALL ASSIGN( TY, 'TY' );
      DriverHandle = Start_PCI_Driver( Driver ) ;
      IF DriverHandle EQ 0(31) THEN
         PUT 'PCI-Driver konnte nicht gestartet werden' TO TY BY SKIP,A;
         PUT 'Terminiert: ', GET_TASKNAME() TO TY BY SKIP,A,A;
         TERMINATE ;
      ELSE
         Basisadresse = FindFirstPCIBase( DriverHandle, Kolter_VendorID, PCI_ADDA_DeviceID, find ) ;
         IF Basisadresse EQ 0(31) THEN
            PUT 'ADDA nicht gefunden' TO TY BY SKIP,A;
            Stop_PCI_Driver( Driver ) ;
            PUT 'Terminiert: ', GET_TASKNAME() TO TY BY SKIP,A,A;
            TERMINATE ;
         ELSE
            Basisadresse = TOFIXED( TOBIT(Basisadresse) AND 'FFFFFFF8'B4 ) ;
            SEMASET 1, semaAD;
            SEMASET 1, semaDA;
            BeendeADDA = '0'B ;
            AD_lsb_spannung = AD_VOLT_RANGE / AD_RESOLUTION ;            
            /* DA-Ausgaenge aus 0. Volt initialisieren */
            CALL DacOut( DACA, 0. ) ;
            CALL DacOut( DACB, 0. ) ;
            CALL DacOut( DACC, 0. ) ;
            CALL DacOut( DACD, 0. ) ;
            
            /* A/D-Wandler auf Ruhestellung setzen:
            ** Kanal 0 selektiert, Read LSB aktiv
            */
            SetPortByte( DriverHandle, Basisadresse + AD_MUX, AD_LetzterKanal ) ;
            SetPortByte( DriverHandle, Basisadresse + AD_RC,  AD_READ_LSB ) ;
            PUT 'PCI-ADDA gefunden auf Adresse ', TOBIT( Basisadresse ) TO TY BY SKIP,A,B4 ;
         FIN ;
      FIN ;
   FIN ;
END; /* INIT_AD PROC ::::::::::::::::::::::::::::::::::::::::::::::::*/



DacOut: PROC( kanal FIXED(31) , spannung FLOAT ) GLOBAL ;
   /*
   ** Ausgabe einer Spannung zwischen -10.000 ... +10.000 Volt 
   */
   DCL (lsb, msb)    FIXED(7) ;
   DCL wandlerwert   BIT(16) ;

   IF    kanal EQ DACA
      OR kanal EQ DACB
      OR kanal EQ DACC
      OR kanal EQ DACD
   THEN
      IF Basisadresse EQ 0(31) THEN
         CALL INIT_ADDA ;
      FIN ;

      IF ABS spannung GT 10. THEN
         spannung = 10. * SIGN(spannung) ;
      FIN ;

      wandlerwert = TOBIT( 2048 + ENTIER(0.5 * 409.5 * spannung ) ) ;
      lsb         = TOFIXED(   wandlerwert              AND '00FF'B4 ) ;
      msb         = TOFIXED( ( wandlerwert SHIFT (-8) ) AND '00FF'B4 ) ;

      REQUEST semaDA ;
         SetPortByte( DriverHandle, Basisadresse + kanal        , lsb ) ; /* setze LSB */
         SetPortByte( DriverHandle, Basisadresse + kanal + 4(31), msb ) ; /* setze MSB */
         lsb = GetPortByte( DriverHandle, Basisadresse + DAC_LATCH ) ;
      RELEASE semaDA ;
   FIN ;
END; /* P DacOut */

ADread: PROC( kanal FIXED ) RETURNS( FLOAT ) GLOBAL ;
   DCL (lsb, msb )   FIXED(7) ;

   IF Basisadresse EQ 0(31) THEN
      CALL INIT_ADDA ;
   FIN ;

   REQUEST semaAD ;
   IF kanal NE AD_LetzterKanal THEN
      SetPortByte( DriverHandle, Basisadresse + AD_MUX, kanal ) ;
      AD_LetzterKanal = kanal ;
      /* Die jetzt erforderliche Einschwingzeit von ca. bis zu 10 us kann schlecht 
      ** hardwareunabhaengig realisiert werden.
      ** Da ein Portzugriff bei einem 33 MHz-PCI ca 1 us erfordert, wird hier der
      ** A/D-Wandler mehrfach ausgelesen.
      */
      lsb = GetPortByte( DriverHandle, Basisadresse + AD_RC ) ;
      lsb = GetPortByte( DriverHandle, Basisadresse + AD_RC ) ;
      lsb = GetPortByte( DriverHandle, Basisadresse + AD_RC ) ;
      lsb = GetPortByte( DriverHandle, Basisadresse + AD_RC ) ;
      lsb = GetPortByte( DriverHandle, Basisadresse + AD_RC ) ;
      lsb = GetPortByte( DriverHandle, Basisadresse + AD_RC ) ;
      lsb = GetPortByte( DriverHandle, Basisadresse + AD_RC ) ;
      lsb = GetPortByte( DriverHandle, Basisadresse + AD_RC ) ;
      lsb = GetPortByte( DriverHandle, Basisadresse + AD_RC ) ;
      lsb = GetPortByte( DriverHandle, Basisadresse + AD_RC ) ;
   FIN ;

   /* Wandlung starten */
   SetPortByte( DriverHandle, Basisadresse + AD_RC, AD_CONVERT ) ;

   /* auf Ready warten */
   lsb = 0(7) ;
   WHILE (lsb//2) * 2 EQ lsb REPEAT ;
      lsb = GetPortByte( DriverHandle, Basisadresse + AD_READY ) ;
   END ;

   /* MSB lesen */
   SetPortByte( DriverHandle, Basisadresse + AD_RC, AD_READ_MSB ) ;
   msb = GetPortByte( DriverHandle, Basisadresse + AD_RC ) ;

   /* LSB lesen */
   SetPortByte( DriverHandle, Basisadresse + AD_RC, AD_READ_LSB ) ;
   lsb = GetPortByte( DriverHandle, Basisadresse + AD_RC ) ;

   RELEASE semaAD ;

   RETURN( ( ( msb * 255(31) + lsb )//16 - AD_WW_NULL ) * AD_lsb_spannung ) ;

END; /* P ADread */

EndADDA: PROC ;
   Basisadresse = 0(31) ; 
   Stop_PCI_Driver( Driver ) ;
END; /* EndADDA */


TestAD: TASK ;
   /*
   ** Fortwhrendes, zyklisches Einlesen der A/D-Kanaele
   */
   DCL (HOME, CNTRZ)   CHAR;
   HOME  = TOCHAR 30;
   CNTRZ = TOCHAR 26;
   CALL INIT_ADDA ;
   PUT CNTRZ TO TY ;
   WHILE NOT BeendeADDA REPEAT ;
      PUT HOME,'Test der AD-Kanaele' TO TY BY SKIP,A,A ;
      FOR i FROM 0 BY 4 TO 15 REPEAT;
         PUT TO TY BY SKIP ;
         FOR j FROM i TO i+3 REPEAT ;
            PUT j,': ',ADread(j),'[V]' TO TY BY F(4),A,F(7,3),A ;
         END; 
      END;
   END ;

   CALL EndADDA ;
   PUT 'Beendet: ', GET_TASKNAME() TO TY BY SKIP,A,A;
END ; /* T TestAD */

TestDA: TASK ;
   /* Testausgaben ueber alle DACS
   ** DAC A: steigender Saegezahn
   ** DAC B: fallender Saegezahn
   ** DAC C: Dreieck
   ** DAC D: Sinus
   */
   CALL INIT_ADDA ;
   PUT 'Test der DA-Kanaele',
       '   DAC A (Pin 1 - 20): steigender Saegezahn',
       '   DAC B (Pin 2 - 21): fallender Saegezahn',
       '   DAC C (Pin 3 - 22): Dreieck',
       '   DAC D (Pin 4 - 23): Sinus'
       TO TY BY SKIP,A ;
   WHILE NOT BeendeADDA REPEAT ;
      FOR i FROM 0 TO 4095 WHILE NOT BeendeADDA REPEAT ;
         CALL DacOut( DACA, i/409.5*2. - 10. ) ;
         CALL DacOut( DACB, (4095-i)/409.5*2. - 10. ) ;
         IF i GE 2047 THEN
            CALL DacOut( DACC, (4095-i)/204.6*2. -10. ) ;
         ELSE
            CALL DacOut( DACC, i/204.6*2.-10. ) ;
         FIN ;
         CALL DacOut( DACD, 10.*SIN(i*2.*PI(1.)/4095.) ) ;
      END;
   END;
   
   CALL EndADDA ;
   PUT 'Beendet: ', GET_TASKNAME() TO TY BY SKIP,A,A;

END; /* T TestDA */

StopADDA: TASK PRIO 10 ;
   BeendeADDA = '1'B ;
END; /* T StopADDA */

Vout_usage: PROC( stdout DATION OUT ALPHIC IDENT ) ;
   PUT  'vout kanal spannung :'
       ,'   Ausgabe einer Spannung auf dem D/A-Teil'
       ,'   kanal:    1...4 entsprechend DAC A ... DAC D'
       ,'   spannung: +10. ... -10. in Volt'
       TO stdout BY  SKIP,A ;
END; /* P Vout_usage */

Vout: PROC( stdin  DATION IN    ALPHIC IDENT,
            stdout DATION   OUT ALPHIC IDENT,
            stderr DATION   OUT ALPHIC IDENT,
            length FIXED,
            argv   CHAR(255)
          ) RETURNS(BIT(1));
   /*
   ** Testausgabe der D/A-Kanaele
   */
   DCL fOK                 BIT(1) INIT( '1'B ) ; ! Annahme: alles ok
   DCL kanal               FIXED ;
   DCL spannung            FLOAT ;
   DCL kan_err             FIXED  INIT( 0 ) ;
   DCL spg_err             FIXED  INIT( 0 ) ;
   DCL basisadr_alt        FIXED(31) ;

   OPEN stdout ; fOK = fOK AND ST(stdout) == 0 ;

   IF fOK THEN
      CONVERT kanal, spannung FROM argv BY RST( kan_err ), F(10), RST( spg_err ), E(20) ;
      fOK = fOK AND kan_err == 0 AND spg_err == 0;

      IF NOT fOK THEN
         IF kan_err NE 0 THEN
            PUT ':: Kanalnummer konnte nicht gelesen werden: ', GET_TASKNAME()
               TO stderr BY SKIP,A,A;
         ELSE
            IF spg_err NE 0 THEN
               PUT ':: Spannung konnte nicht gelesen werden: ', GET_TASKNAME()
                  TO stderr BY SKIP,A,A;
            FIN ;
         FIN ;
         PUT ':: Parameter error: ', GET_TASKNAME()
               TO stderr BY SKIP,A,A ;
      ELSE
         IF kanal LE 0 OR kanal GT 4 THEN
            PUT ':: Kanal ', kanal, 'ungueltig! Nur Kanaele 1...4 unterstuetzt: ', GET_TASKNAME()
               TO stderr BY SKIP,A,F(3),A,A;
            fOK = '0'B ;
         ELSE
            IF spannung GT 10. OR spannung LT -10. THEN
               PUT ':: Spannung ', spannung, ' [V] nicht im erlaubten Wertebereich (-10. ... +10.): ', GET_TASKNAME()
                  TO stderr BY SKIP,A,F(7,3),A,A;
               fOK = '0'B ;
            ELSE
               basisadr_alt = Basisadresse ;
               IF basisadr_alt == 0(31) THEN
                  CALL INIT_ADDA ;
               FIN ;
               PUT 'Ausgabe ', spannung, ' [V] auf Kanal ', kanal
                  TO stdout BY SKIP,A,F(7,3),A,F(3);
               CASE kanal
                  ALT (1) CALL DacOut( DACA, spannung ) ;
                  ALT (2) CALL DacOut( DACB, spannung ) ;
                  ALT (3) CALL DacOut( DACC, spannung ) ;
                  ALT (4) CALL DacOut( DACD, spannung ) ;
               FIN ;
               IF basisadr_alt == 0(31) THEN
                  CALL EndADDA ; /* Treiber freigeben */
               FIN ;
            FIN ;
         FIN ;
      FIN ;

      PUT 'Ende ',GET_TASKNAME() TO stdout BY SKIP,A,A ;
   ELSE
      PUT ':: open stdout failed: ', GET_TASKNAME()
         TO stderr BY SKIP,A,A;
   FIN ;

   CLOSE stdout ;

   RETURN( fOK ) ;

END; /* P Vout */

InitAdda: TASK ;
   CALL INIT_ADDA ;
END; /* T InitAdda */

Tick: TASK RESIDENT;
   DCL f7 FIXED(7) ;
   f7 = f7 + 1 ;
   SetPortByte( DriverHandle, Basisadresse + DIG_OUT, f7 ) ; 
END; /* T Tick */

Square: TASK ;
   DCL f7 FIXED(7) ;
   /* Ausgabe eines Rechtecks ueber den Digitalausgang */
   CALL INIT_ADDA ;
   PUT 'Ausgabe eines Rechtecks ueber den Digitalausgang'
       TO TY BY SKIP,A ;
   WHILE NOT BeendeADDA REPEAT ;
      f7 = f7 + 1 ;
      SetPortByte( DriverHandle, Basisadresse + DIG_OUT, f7 ) ; 
   END;
   CALL EndADDA ;
   PUT 'Beendet: ', GET_TASKNAME() TO TY BY SKIP,A,A;
END; /* T Square */

MODEND ; /* M ADDA */